<?php

namespace XF\Cli\Command\Designer;

use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use XF\Cli\Command\AbstractCommand;
use XF\Cli\Command\StyleArchiveTrait;
use XF\Entity\Template;
use XF\Finder\TemplateFinder;
use XF\Finder\TemplateMapFinder;

class TouchTemplate extends AbstractCommand
{
	use RequiresDesignerModeTrait;
	use StyleArchiveTrait;

	protected function configure()
	{
		$this
			->setName('xf-designer:touch-template')
			->setDescription('Marks the specified template as modified in the specified style and exports it.')
			->addArgument(
				'designer-mode',
				InputArgument::REQUIRED,
				'Designer mode ID'
			)
			->addArgument(
				'template',
				InputArgument::REQUIRED,
				'Template to mark as modified. Must include the type prefix, e.g. \'public:template_name\''
			)
			->addOption(
				'custom',
				'c',
				InputOption::VALUE_NONE,
				'If specified, allows the creation of a custom template.'
			);
	}

	protected function execute(InputInterface $input, OutputInterface $output)
	{
		$em = \XF::em();

		$style = $this->getStyleByDesignerModeInput($input, $output);

		$templater = \XF::app()->templater();

		$title = $input->getArgument('template');
		[$type, $title] = $templater->getTemplateTypeAndName($title);

		if (!$type)
		{
			$type = 'public';
		}

		if (!$input->getOption('custom'))
		{
			$templateFinder = $em->getFinder(TemplateFinder::class)
				->where([
					'type' => $type,
					'title' => $title,
				]);

			if (!$templateFinder->total())
			{
				$output->writeln("The template '$type:$title' does not exist in any style. Pass the --custom option to create it.");
				return 1;
			}
		}

		$templateMap = $em->getFinder(TemplateMapFinder::class)
			->where([
				'style_id' => $style->style_id,
				'type' => $type,
				'title' => $title,
			])
			->with('Template', true)
			->fetchOne();

		if (!$templateMap)
		{
			// entirely custom template
			$newTemplate = $em->create(Template::class);
			$newTemplate->style_id = $style->style_id;
			$newTemplate->type = $type;
			$newTemplate->title = $title;
			$newTemplate->template = '';
			$newTemplate->addon_id = '';

			if (!$newTemplate->preSave())
			{
				$output->writeln($newTemplate->getErrors());
				return 1;
			}
			$newTemplate->save();
		}
		else
		{
			$baseTemplate = $templateMap->Template;

			if ($baseTemplate->style_id == $style->style_id)
			{
				// template already exists in this style
				$output->writeln(["", "The template '$type:$title' has already been modified. Run the 'xf-designer:export-templates' command to export."]);
				return 1;
			}
			else
			{
				// template only exists in a parent; duplicate it here
				$newTemplate = $em->create(Template::class);
				$newTemplate->style_id = $style->style_id;
				$newTemplate->type = $baseTemplate->type;
				$newTemplate->title = $baseTemplate->title;
				$newTemplate->template = $baseTemplate->template;
				$newTemplate->addon_id = $baseTemplate->addon_id;

				if (!$newTemplate->preSave())
				{
					$output->writeln($newTemplate->getErrors());
					return 1;
				}
				$newTemplate->save();
			}
		}

		$output->writeln(["", "Template '{$newTemplate->type}:{$newTemplate->title}' modified in style '$style->title'", ""]);

		return 0;
	}
}
